* Title: 	table10.do
* Version: 	23 May 2018
* Purpose: 	Produce elasticity estimates

*******************************************************************************
* (0) Start of file
*******************************************************************************

capture log close
log using output/tables/log/table10, text replace
set more off
clear all
macro drop _all


*******************************************************************************
* (1) Load data
*******************************************************************************

use year statefip conspuma UnempB PrimeMaleD EducationI incwage_real_emp perwt ///
	using dta/census_clean, clear

* Keep only prime-age men
keep if PrimeMaleD == 1

* Keep only continental 48 states (excluding District of Columbia)
drop if (statefip == 02 | statefip == 11 | statefip == 15 | statefip>56)

* Keep only 1980/1990/2000 Census and 2009-2011 ACS
keep if year == 1980 | year == 1990 | year == 2000 | (year>=2009 & year<=2011)

* Pool data for 2009-2011 ACS
replace year = 2010 if year>=2009 & year<=2011

* Fix coding due to post-2006 PUMA change due to Katrina
replace conspuma = 172 if conspuma == 173


*******************************************************************************
* (2) Create PUMA-level variables
*******************************************************************************

* Create less than HS/college education variables
gen LessHSEd 	= (EducationI == 1)
gen CollegeEd 	= (EducationI == 4)

gen double log_incwage_real_emp = log(incwage_real_emp)

collapse	(sum)   Total = PrimeMaleD Jobless = UnempB LessHSEd CollegeEd ///
			(p25)   log_incwage_p25 = log_incwage_real_emp ///
			[pw=perwt], by(conspuma statefip year)

gen double Jobless_Rate = Jobless / Total
gen double College_Rate = CollegeEd / Total
gen double LessHSEd_Rate = LessHSEd / Total

* Generate employment rates
gen double Emp_Rate = 1 - Jobless_Rate
gen double LogEmpRate = log(Emp_Rate)

preserve
keep if year==1980
foreach var of varlist Jobless_Rate College_Rate LessHSEd_Rate {
	rename `var' `var'_1980
}
keep conspuma *_1980
tempfile 1980
save `1980'

restore
merge m:1 conspuma using `1980', assert(matched) nogen


*******************************************************************************
* (3) Merge with Bartik shocks
*******************************************************************************

merge 1:1 year conspuma using dta/bartik_census, assert(matched) nogen

* Generate interactions with 1980 jobless rate
gen double IncWage_p25_IntJob = log_incwage_p25 * Jobless_Rate_1980
gen double Bartik_IntJob = Bartik * Jobless_Rate_1980

* Generate polynomial instrument terms
gen double Bartik_Sq = Bartik^2
gen double Bartik_Cu = Bartik^3

gen double Bartik_IntJob_Sq = Bartik_IntJob^2
gen double Bartik_IntJob_Cu = Bartik_IntJob^3

* Keep years with Bartik shock
keep if year >= 1990

* Save dataset
save output/tables/dta/table10.dta, replace


*******************************************************************************
* (4) Regressions
*******************************************************************************

* Column 1: OLS, excluding state fixed effects
reg LogEmpRate log_incwage_p25 IncWage_p25_IntJob Jobless_Rate_1980 i.year College_Rate_1980 LessHSEd_Rate_1980, vce(cluster statefip)

* Column 2: IV, excluding state fixed effects
ivreg2 LogEmpRate (log_incwage_p25 IncWage_p25_IntJob = Bartik Bartik_Sq Bartik_Cu Bartik_IntJob Bartik_IntJob_Sq Bartik_IntJob_Cu) ///
					Jobless_Rate_1980 i.year College_Rate_1980 LessHSEd_Rate_1980, ///
					first cluster(statefip)

* Column 3: OLS, including state fixed effects
reg LogEmpRate log_incwage_p25 IncWage_p25_IntJob Jobless_Rate_1980 i.year College_Rate_1980 LessHSEd_Rate_1980 i.statefip, vce(cluster statefip)

* Column 4: IV, including state fixed effects
ivreg2 LogEmpRate (log_incwage_p25 IncWage_p25_IntJob = Bartik Bartik_Sq Bartik_Cu Bartik_IntJob Bartik_IntJob_Sq Bartik_IntJob_Cu) ///
					Jobless_Rate_1980 College_Rate_1980 LessHSEd_Rate_1980 i.statefip i.year, ///
					first partial(i.statefip i.year) cluster(statefip)


*******************************************************************************
* (5) End of file
*******************************************************************************

log close
exit, clear
